# One-Way ANOVA

One-Way ANOVA tests whether **three or more independent groups** differ in their mean values on a continuous outcome.

---

# 📌 When to Use
Use One-Way ANOVA when:
- You have **one categorical factor** with 3 or more levels  
- Your dependent variable is **continuous**  
- Groups are **independent**  
- Variances are reasonably equal

---

# 📐 Simple Formula

The F-statistic for One-Way ANOVA is:

$$
F = \frac{MS_{between}}{MS_{within}}
$$

Where:

- $MS_{between}$ = Mean Square Between Groups  
- $MS_{within}$  = Mean Square Within Groups  

These are computed from the **Sum of Squares (SS)** components:

### Sum of Squares Definitions

- $SS_{between}$ (Between-Groups Sum of Squares):  
  Measures variability **due to differences between group means**.  
  Computed as the weighted squared distance between each group mean and the overall mean.

- $SS_{within}$ (Within-Groups Sum of Squares):  
  Measures variability **inside groups**, due to individual differences.  
  Computed as the sum of squared deviations of each observation from its own group mean.

### Converting SS into MS

$$
MS_{between} = \frac{SS_{between}}{k - 1}
$$

$$
MS_{within} = \frac{SS_{within}}{N - k}
$$

Where:

- $k$ = number of groups  
- $N$ = total number of observations  

Larger $MS_{between}$ relative to $MS_{within}$ means the groups differ more strongly, leading to a larger F value.

---

# 🔬 Numerical Example (Simplified)

Group means:
- Group A: 5  
- Group B: 10  
- Group C: 12  

Between-group variance large compared to within-group variance → large $F$ → likely significant.

EasyStatistics computes the full ANOVA table automatically.

---

# 🧩 Post-Hoc Tests (Automatically Done)
If the ANOVA is significant, EasyStatistics performs:

- **Tukey HSD**  
- **Bonferroni correction**  
- Group-by-group comparisons  
- Effect sizes  

---

# 🧪 Assumptions
- Normality within groups  
- Homogeneity of variances  
- Independent observations  

---

# 🧠 Interpretation
A significant $F$ indicates **at least one** group differs in mean.  
Post-hoc tests show **which groups** differ.

---

# 🐍 Python Implementation

Statsmodels:
https://www.statsmodels.org/stable/anova.html

```python
import statsmodels.api as sm
from statsmodels.formula.api import ols

model = ols('Y ~ C(group)', data=df).fit()
anova_table = sm.stats.anova_lm(model, typ=2)
print(anova_table)
```
# 📎 Related Tests

[Two-Way ANOVA](/statistics/anova/two-way-anova.md)

[Kruskal–Wallis Test](/statistics/anova/kruskal-wallis.md)

[ANCOVA](/statistics/anova/ancova.md)

[MANOVA](/statistics/anova/manova.md)
